<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\User_plans;
use App\Models\Plans;
use App\Models\Tp_Transaction;
use App\Models\Settings;
use App\Mail\NewRoi;
use App\Mail\endplan;
use Carbon\Carbon;
use Mail;

class AutoRoiCommand extends Command
{
    protected $signature = 'roi:process';
    protected $description = 'Automatically process ROI for active user plans';

    public function handle()
    {
        $settings = Settings::first();

        if ($settings->trade_mode !== 'on') {
            return;
        }

        $now = now();

        $usersPlans = User_plans::where('active', 'yes')->get();

        foreach ($usersPlans as $plan) {
            $dplan = Plans::find($plan->plan);
            $user = User::find($plan->user);
              \Log::info("Processing plan ID {$plan->id} for user ID {$plan->user}");
            
              switch ($dplan->increment_interval) {
                case "Monthly":
                    $nextDrop = $plan->last_growth->copy()->addDays(25);
                    break;
                case "Weekly":
                    $nextDrop = $plan->last_growth->copy()->addDays(6);
                    break;
                case "Daily":
                    $nextDrop = $plan->last_growth->copy()->addHours(20);
                    break;
                case "Hourly":
                    $nextDrop = $plan->last_growth->copy()->addMinutes(49);
                    break;
                case "Every 30 Minutes":
                    $nextDrop = $plan->last_growth->copy()->addMinutes(19);
                    break;
                default:
                    $nextDrop = $plan->last_growth->copy()->addMinutes(4);
                    break;
            }

           
            $isActive = $now->lessThanOrEqualTo($plan->expire_date) && $user->trade_mode === 'on';
            $isExpired = $now->greaterThan($plan->expire_date);

            $increment = $dplan->increment_type === "Percentage"
                ? ($plan->amount * $dplan->increment_amount) / 100
                : $dplan->increment_amount;

            if ($isActive) {
                $canTradeNow = $now->isWeekday() || $settings->weekend_trade === 'on';
                
                  \Log::info("Processing ROI for user {$user->id}. Increment: {$increment}");

                if (
                    $canTradeNow &&
                    $now->greaterThanOrEqualTo($nextDrop->copy()->subMinutes(5)) && // 5-minute window
                    $now->lessThanOrEqualTo($nextDrop)
                ) {
                    User::where('id', $user->id)->update([
                        'roi' => $user->roi + $increment,
                        'account_bal' => $user->account_bal + $increment,
                    ]);

                    Tp_Transaction::create([
                        'plan' => $dplan->name,
                        'user' => $user->id,
                        'amount' => $increment,
                        'user_plan_id' => $plan->id,
                        'type' => 'ROI',
                    ]);

                    $plan->update([
                        'last_growth' => $nextDrop,
                        'profit_earned' => $plan->profit_earned + $increment,
                    ]);

                    if ($user->sendroiemail === 'Yes') {
                        Mail::to($user->email)->send(new NewRoi(
                            $user,
                            $dplan->name,
                            $increment,
                            $now->toDateTimeString(),
                            'New Return on Investment(ROI)'
                        ));
                    }
                }

                // Still update growth on weekends if weekend trade is off
                if ($now->isWeekend() && $settings->weekend_trade !== 'on' && $now->greaterThanOrEqualTo($plan->last_growth)) {
                    $plan->update([
                        'last_growth' => $nextDrop,
                    ]);
                }
            }

            // Expired plan handling
            if ($isExpired) {
                if ($settings->return_capital) {
                    User::where('id', $user->id)->update([
                        'account_bal' => $user->account_bal + $plan->amount,
                    ]);
                    
                     \Log::info("Plan ID {$plan->id} has expired for user {$user->id}.");

                    Tp_Transaction::create([
                        'plan' => $dplan->name,
                        'user' => $user->id,
                        'amount' => $plan->amount,
                        'type' => 'Investment capital',
                    ]);
                }

                $plan->update([
                    'active' => 'expired',
                ]);

                if ($user->sendinvplanemail === 'Yes') {
                    $objDemo = (object) [
                        'receiver_email' => $user->email,
                        'receiver_plan' => $dplan->name,
                        'received_amount' => "{$settings->currency}{$plan->amount}",
                        'sender' => $settings->site_name,
                        'receiver_name' => $user->name,
                        'date' => $now,
                        'subject' => "Investment plan closed"
                    ];

                    Mail::to($user->email)->send(new endplan($objDemo));
                }
            }
            \Log::info("AutoRoiCommand finished processing.");
        }
    }
}
